import 'dotenv/config';
import express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import compression from 'compression';
import cookieParser from 'cookie-parser';
import rateLimit from 'express-rate-limit';
import http from 'http';
import path from 'path';
import { fileURLToPath } from 'url';
import fs from 'fs';
import authRouter from './routes/auth.js';
import walletsRouter from './routes/wallets.js';
import webhooksRouter from './routes/webhooks.js';
import adminRouter from './routes/admin.js';
import systemRouter from './routes/system.js';
import publicRouter from './routes/public.js';
import kycRouter from './routes/kyc.js';
import notificationsRouter from './routes/notifications.js';
import supportRouter from './routes/support.js';
import paymentsRouter from './routes/payments.js';
import tradeRouter from './routes/trade.js';
import { initIO } from './socket.js';
import { startMarketFeed } from './services/marketFeed.js';
const app = express();
const server = http.createServer(app);
const CLIENT_ORIGIN = process.env.CLIENT_ORIGIN || 'http://localhost:3000';
const io = initIO(server, CLIENT_ORIGIN);
// Trust proxy when behind a load balancer/reverse proxy
if (String(process.env.TRUST_PROXY || '').trim() === '1') {
    app.set('trust proxy', 1);
}
// Hide implementation header
app.disable('x-powered-by');
app.use(cors({ origin: CLIENT_ORIGIN, credentials: true }));
// Allow cross-origin resource loading (images) from 3000
app.use(helmet({ crossOriginResourcePolicy: { policy: 'cross-origin' } }));
app.use(compression());
app.use(express.json({ limit: '2mb' }));
app.use(cookieParser());
// Configurable rate limiter (disabled in development by default)
const RL_WINDOW_MS = Number(process.env.RATE_LIMIT_WINDOW_MS || 15 * 60 * 1000);
const RL_MAX = Number(process.env.RATE_LIMIT_MAX || 300);
const RL_DISABLED = String(process.env.RATE_LIMIT_DISABLED || '').trim() === '1' || process.env.NODE_ENV !== 'production';
app.use(rateLimit({
    windowMs: RL_WINDOW_MS,
    max: RL_MAX,
    standardHeaders: true,
    legacyHeaders: false,
    message: 'Too many requests, please try again later.',
    skip: () => RL_DISABLED,
}));
// Serve local assets (coin logos, etc.)
const __dirname = path.dirname(fileURLToPath(import.meta.url));
app.use('/assets', express.static(path.join(__dirname, '..', 'assets')));
app.get('/health', (_req, res) => { res.json({ ok: true, timestamp: Date.now() }); });
app.use('/api/auth', authRouter);
app.use('/api/wallets', walletsRouter);
app.use('/api/webhooks', webhooksRouter);
app.use('/api/admin', adminRouter);
app.use('/api/system', systemRouter);
app.use('/api/public', publicRouter);
app.use('/api/kyc', kycRouter);
app.use('/api/notifications', notificationsRouter);
app.use('/api/support', supportRouter);
app.use('/api/payments', paymentsRouter);
app.use('/api/trade', tradeRouter);
// Serve built frontend in production (or when explicitly enabled)
const SERVE_CLIENT = String(process.env.SERVE_CLIENT || '').trim() === '1' || process.env.NODE_ENV === 'production';
if (SERVE_CLIENT) {
    const candidateDir = process.env.CLIENT_BUILD_DIR || process.env.STATIC_DIR || path.join(__dirname, '..', '..', 'crypto-wallet', 'build');
    if (fs.existsSync(candidateDir)) {
        app.use(express.static(candidateDir));
        // SPA fallback (exclude API, assets, and health)
        app.get(/^(?!\/api\/|\/assets\/|\/health).*/, (_req, res) => {
            res.sendFile(path.join(candidateDir, 'index.html'));
        });
    }
}
io.on('connection', (socket) => {
    console.log('socket connected', socket.id);
    socket.on('disconnect', () => console.log('socket disconnected', socket.id));
});
// Start market feed with defaults/env
const MARKET_FEED_VS = (process.env.MARKET_FEED_VS || 'USD').split(',').map(s => s.trim()).filter(Boolean);
const MARKET_FEED_INTERVAL_MS = Number(process.env.MARKET_FEED_INTERVAL_MS || 30000);
startMarketFeed(io, { defaultVs: MARKET_FEED_VS, intervalMs: MARKET_FEED_INTERVAL_MS });
const PORT = Number(process.env.PORT || 4000);
const RAW_HOST = String(process.env.HOST || '0.0.0.0').trim();
// Allow accidental protocol in HOST; strip scheme and any path
const HOST = RAW_HOST.replace(/^https?:\/\//i, '').split('/')[0];
server.listen(PORT, HOST, () => console.log(`API listening on http://${HOST}:${PORT}`));
