import { Router } from 'express';
import { z } from 'zod';
import bcrypt from 'bcryptjs';
import pool from '../config/db.js';
import { authGuard } from '../middleware/auth.js';
import { roleGuard } from '../middleware/roles.js';
import crypto from 'crypto';
import { getOnlineUserIds, emitToUser } from '../socket.js';
import { getIO } from '../socket.js';
import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { getSimplePrices } from '../services/coingecko.js';
const router = Router();
const __dirname = path.dirname(fileURLToPath(import.meta.url));
const kycBaseDir = path.join(__dirname, '..', '..', 'assets', 'kyc');
async function insertNotification(userId, payload) {
    await pool.query(`INSERT INTO notifications (user_id, type, title, message, level, data) VALUES (:userId, :type, :title, :message, :level, :data)`, { userId, type: payload.type, title: payload.title, message: payload.message || null, level: payload.level || 'info', data: payload.data ? JSON.stringify(payload.data) : null });
}
// Presence list (ids online)
router.get('/presence', authGuard, roleGuard([1, 2]), async (_req, res) => {
    res.json({ onlineUserIds: getOnlineUserIds() });
});
// Pending transactions count (admin/agent)
router.get('/transactions/pending-count', authGuard, roleGuard([1, 2]), async (req, res) => {
    const user = req.user;
    if (user.role === 1) {
        const [rows] = await pool.query(`SELECT COUNT(*) AS cnt FROM transactions WHERE status = 'pending'`);
        const cnt = rows && rows.length ? Number(rows[0].cnt) : 0;
        return res.json({ count: cnt });
    }
    // Agent: count only owned users' pending transactions
    const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!me || !me.length)
        return res.status(401).json({ error: 'Unauthorized' });
    const myId = me[0].id;
    const [rows] = await pool.query(`SELECT COUNT(*) AS cnt FROM transactions t JOIN users u ON u.id = t.user_id WHERE t.status = 'pending' AND u.owner_id = :ownerId`, { ownerId: myId });
    const cnt = rows && rows.length ? Number(rows[0].cnt) : 0;
    return res.json({ count: cnt });
});
// Referral link management (admin/agent can have one code)
router.post('/referral', authGuard, roleGuard([1, 2]), async (req, res) => {
    const user = req.user;
    const code = crypto.randomBytes(12).toString('hex');
    try {
        // Map uid to numeric id
        const [rows] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
        if (!rows || !rows.length)
            return res.status(400).json({ error: 'Owner not found' });
        const ownerNumericId = rows[0].id;
        await pool.query(`INSERT INTO referral_codes (owner_user_id, code) VALUES (:uid, :code)
       ON DUPLICATE KEY UPDATE code = VALUES(code)`, { uid: ownerNumericId, code });
        return res.json({ code });
    }
    catch (e) {
        return res.status(500).json({ error: 'Failed to create referral link' });
    }
});
router.get('/referral', authGuard, roleGuard([1, 2]), async (req, res) => {
    const user = req.user;
    const [owner] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!owner || !owner.length)
        return res.json({ code: null });
    const [rows] = await pool.query(`SELECT code FROM referral_codes WHERE owner_user_id = :uid LIMIT 1`, { uid: owner[0].id });
    if (!rows || rows.length === 0)
        return res.json({ code: null });
    return res.json({ code: rows[0].code });
});
// List users (admin/agent)
router.get('/users', authGuard, roleGuard([1, 2]), async (req, res) => {
    const user = req.user;
    if (user.role === 1) {
        const [rows] = await pool.query(`SELECT u.id, u.uid, u.email, u.first_name, u.last_name, u.role_id, u.is_verified, u.is_email_verified, u.last_login_at, u.owner_id,
              (SELECT COUNT(*) FROM transactions t WHERE t.user_id = u.id AND t.status = 'pending') AS pending_tx_count,
              (SELECT COUNT(*) FROM support_tickets st WHERE st.user_id = u.id AND st.status = 'open') AS pending_tickets_count,
              (SELECT COUNT(*) FROM orders o WHERE o.user_id = u.id AND o.status = 'pending') AS pending_orders_count
       FROM users u ORDER BY u.id DESC LIMIT 500`);
        return res.json(rows);
    }
    const [owner] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!owner || !owner.length)
        return res.json([]);
    console.log('Agent query - owner ID:', owner[0].id, 'agent uid:', user.sub);
    const [rows] = await pool.query(`SELECT u.id, u.uid, u.email, u.first_name, u.last_name, u.role_id, u.is_verified, u.is_email_verified, u.last_login_at, u.owner_id,
            (SELECT COUNT(*) FROM transactions t WHERE t.user_id = u.id AND t.status = 'pending') AS pending_tx_count,
            (SELECT COUNT(*) FROM support_tickets st WHERE st.user_id = u.id AND st.status = 'open') AS pending_tickets_count,
            (SELECT COUNT(*) FROM orders o WHERE o.user_id = u.id AND o.status = 'pending') AS pending_orders_count
     FROM users u 
     WHERE u.role_id = 3 AND (u.owner_id = :ownerId OR u.owner_id IS NULL)
     ORDER BY u.id DESC LIMIT 500`, { ownerId: owner[0].id });
    console.log('Agent query - found users:', rows.length, 'with owner_id =', owner[0].id);
    console.log('Sample users:', rows.slice(0, 3).map(u => ({ id: u.id, email: u.email, owner_id: u.owner_id })));
    // Debug: Check all users with owners
    const [allOwnedUsers] = await pool.query(`SELECT u.id, u.email, u.owner_id, o.email as owner_email 
     FROM users u 
     LEFT JOIN users o ON o.id = u.owner_id 
     WHERE u.owner_id IS NOT NULL 
     ORDER BY u.owner_id, u.id 
     LIMIT 10`);
    console.log('All owned users in system:', allOwnedUsers);
    return res.json(rows);
});
// Get user profile by uid (admin/agent)
router.get('/users/:uid', authGuard, roleGuard([1, 2]), async (req, res) => {
    const { uid } = req.params;
    const [rows] = await pool.query(`SELECT u.id, u.uid, u.email, u.first_name, u.last_name, u.role_id, u.is_verified, u.is_email_verified, u.is_disabled, u.last_login_at, u.last_login_ip, u.last_login_ua, u.created_at,
            u.owner_id, o.email AS owner_email
     FROM users u
     LEFT JOIN users o ON o.id = u.owner_id
     WHERE u.uid = :uid LIMIT 1`, { uid });
    if (!rows || rows.length === 0)
        return res.status(404).json({ error: 'Not found' });
    return res.json(rows[0]);
});
// Update user profile (admin/agent)
const updateUserSchema = z.object({
    firstName: z.string().min(1).optional(),
    lastName: z.string().min(1).optional(),
    email: z.string().email().optional(),
});
router.patch('/users/:uid', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid } = req.params;
    const parsed = updateUserSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input', issues: parsed.error.issues });
    const [rows] = await pool.query(`SELECT id, owner_id, email FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'User not found' });
    const target = rows[0];
    // Agent: only update owned users
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        if (Number(target.owner_id) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    const updates = [];
    const params = { id: target.id };
    if (typeof parsed.data.firstName === 'string') {
        updates.push('first_name = :firstName');
        params.firstName = parsed.data.firstName;
    }
    if (typeof parsed.data.lastName === 'string') {
        updates.push('last_name = :lastName');
        params.lastName = parsed.data.lastName;
    }
    if (typeof parsed.data.email === 'string' && parsed.data.email !== target.email) {
        const [exists] = await pool.query(`SELECT id FROM users WHERE email = :email AND id <> :id LIMIT 1`, { email: parsed.data.email, id: target.id });
        if (exists && exists.length)
            return res.status(409).json({ error: 'Email already in use' });
        updates.push('email = :email');
        params.email = parsed.data.email;
    }
    if (updates.length === 0)
        return res.json({ ok: true });
    await pool.query(`UPDATE users SET ${updates.join(', ')} WHERE id = :id`, params);
    return res.json({ ok: true });
});
// KYC approve/reject for a user (admin/agent)
const kycSchema = z.object({ status: z.enum(['accept', 'reject']), reason: z.string().optional() });
router.patch('/users/:uid/kyc', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid } = req.params;
    const parsed = kycSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input', issues: parsed.error.issues });
    // Resolve target user and ownership
    const [rows] = await pool.query(`SELECT id, email, owner_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || rows.length === 0)
        return res.status(404).json({ error: 'User not found' });
    const target = rows[0];
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        if (Number(target.owner_id) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    const isVerified = parsed.data.status === 'accept' ? 1 : 0;
    await pool.query(`UPDATE users SET is_verified = :isVerified WHERE uid = :uid`, { isVerified, uid });
    try {
        if (parsed.data.status === 'accept') {
            // Persist + notify client
            await insertNotification(target.id, { type: 'kyc', title: 'KYC Approved', message: 'Your KYC has been approved', level: 'success', data: { uid } });
            emitToUser(uid, 'kyc:accepted', { userId: uid, email: target.email });
        }
        else {
            await insertNotification(target.id, { type: 'kyc', title: 'KYC Rejected', message: parsed.data.reason || 'Your KYC was rejected', level: 'warning', data: { uid, reason: parsed.data.reason } });
            emitToUser(uid, 'kyc:rejected', { userId: uid, email: target.email, reason: parsed.data.reason });
        }
    }
    catch { }
    return res.json({ ok: true, isVerified: !!isVerified });
});
// List KYC documents for a user (admin/agent) with categories
router.get('/users/:uid/kyc/docs', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid } = req.params;
    // Resolve target user and ownership
    const [rows] = await pool.query(`SELECT id, email, owner_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'User not found' });
    const target = rows[0];
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        if (Number(target.owner_id) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    const userDir = path.join(kycBaseDir, uid);
    const categories = ['idDocument', 'selfie', 'proofOfAddress'];
    const listByCat = {};
    for (const cat of categories) {
        const dir = path.join(userDir, cat);
        let arr = [];
        try {
            const names = fs.readdirSync(dir);
            arr = names.map((name) => {
                const fpath = path.join(dir, name);
                const stat = fs.statSync(fpath);
                const tsFromName = Number(name.split('-')[0]);
                const uploadedAt = isFinite(tsFromName) ? new Date(tsFromName).toISOString() : stat.mtime.toISOString();
                return { name, url: `/assets/kyc/${uid}/${cat}/${name}`, size: stat.size, uploadedAt };
            });
        }
        catch {
            arr = [];
        }
        listByCat[cat] = arr;
    }
    res.json({ categories: listByCat });
});
// Get wallets for user by uid (admin/agent)
router.get('/users/:uid/wallets', authGuard, roleGuard([1, 2]), async (req, res) => {
    const { uid } = req.params;
    const [u] = await pool.query(`SELECT id, JSON_UNQUOTE(JSON_EXTRACT(preferences, '$.currency')) as prefCurrency FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!u || !u.length)
        return res.status(404).json({ error: 'Not found' });
    const userId = u[0].id;
    const vsCurrency = String(u[0].prefCurrency || 'USD').toUpperCase();
    const [rows] = await pool.query(`SELECT uw.id, uw.coin_id, uw.balance, uw.locked, c.symbol, c.name, c.decimals, c.logo_url, c.coingecko_id as coingeckoId,
            (
              SELECT wa.address FROM wallet_addresses wa
              WHERE wa.user_wallet_id = uw.id AND wa.is_active = 1 AND wa.kind = 'deposit'
              ORDER BY wa.id DESC LIMIT 1
            ) AS address,
            (
              SELECT wa.tag_memo FROM wallet_addresses wa
              WHERE wa.user_wallet_id = uw.id AND wa.is_active = 1 AND wa.kind = 'deposit'
              ORDER BY wa.id DESC LIMIT 1
            ) AS tag_memo
     FROM user_wallets uw
     JOIN coins c ON c.id = uw.coin_id
     WHERE uw.user_id = :userId
     ORDER BY c.id ASC`, { userId });
    const ids = Array.from(new Set(rows.map(r => String(r.coingeckoId || '').toLowerCase()).filter(Boolean)));
    let prices = {};
    if (ids.length > 0) {
        try {
            prices = await getSimplePrices(ids, vsCurrency, { ttlMs: 60000 });
        }
        catch { }
    }
    return res.json(rows.map(r => ({
        ...r,
        price: r.coingeckoId ? (prices[String(r.coingeckoId).toLowerCase()] ?? null) : null,
        currency: vsCurrency,
    })));
});
// Get transactions for user by uid (admin/agent)
router.get('/users/:uid/transactions', authGuard, roleGuard([1, 2]), async (req, res) => {
    const { uid } = req.params;
    const limit = Math.min(Number(req.query.limit || 20), 100);
    const offset = Math.max(Number(req.query.offset || 0), 0);
    const [u] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!u || !u.length)
        return res.status(404).json({ error: 'Not found' });
    const userId = u[0].id;
    const [rows] = await pool.query(`SELECT t.id, t.public_id, t.tx_type, t.source, t.status, t.amount, t.fee, t.from_address, t.to_address, t.tx_hash, t.network, t.confirmations, t.created_at, c.symbol, c.logo_url, t.note
     FROM transactions t
     LEFT JOIN coins c ON c.id = t.coin_id
     WHERE t.user_id = :userId
     ORDER BY t.created_at DESC
     LIMIT :limit OFFSET :offset`, { userId, limit, offset });
    return res.json({ items: rows, limit, offset });
});
// System transactions list (admin sees all, agent sees only owned users)
router.get('/transactions', authGuard, roleGuard([1, 2]), async (req, res) => {
    const user = req.user;
    const uidFilter = String(req.query.uid || '');
    const limit = Math.min(Number(req.query.limit || 20), 100);
    const offset = Math.max(Number(req.query.offset || 0), 0);
    // Resolve current account's numeric id
    const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!me || !me.length)
        return res.status(401).json({ error: 'Unauthorized' });
    const myId = me[0].id;
    const params = { limit, offset };
    let where = '1=1';
    if (uidFilter) {
        const [u] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: uidFilter });
        if (!u || !u.length)
            return res.json({ items: [], limit, offset });
        params.userId = u[0].id;
        where += ' AND t.user_id = :userId';
    }
    // Agent ownership restriction
    if (user.role === 2) {
        params.ownerId = myId;
        where += ' AND u.owner_id = :ownerId';
    }
    const [rows] = await pool.query(`SELECT t.id, t.public_id, t.tx_type, t.source, t.status, t.amount, t.fee, t.from_address, t.to_address, t.tx_hash, t.network,
            t.confirmations, t.created_at,
            c.symbol, c.logo_url,
            u.uid as userUid, u.email as userEmail,
            t.note
     FROM transactions t
     JOIN users u ON u.id = t.user_id
     LEFT JOIN coins c ON c.id = t.coin_id
     WHERE ${where}
     ORDER BY t.created_at DESC
     LIMIT :limit OFFSET :offset`, params);
    return res.json({ items: rows, limit, offset });
});
// Create user (admin)
const createUserSchema = z.object({
    firstName: z.string().min(1),
    lastName: z.string().min(1),
    email: z.string().email(),
    password: z.string().min(8),
    roleId: z.number().int().min(1).max(3).default(3),
});
router.post('/users', authGuard, roleGuard([1]), async (req, res) => {
    const parsed = createUserSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input', issues: parsed.error.issues });
    const { firstName, lastName, email, password, roleId } = parsed.data;
    const conn = await pool.getConnection();
    try {
        const [existing] = await conn.query(`SELECT id FROM users WHERE email = :email LIMIT 1`, { email });
        if (existing && existing.length)
            return res.status(409).json({ error: 'Email already registered' });
        const hash = await bcrypt.hash(password, 10);
        const [result] = await conn.query(`INSERT INTO users (role_id, email, password_hash, first_name, last_name) 
       VALUES (:roleId, :email, :hash, :firstName, :lastName)`, { roleId, email, hash, firstName, lastName });
        const userId = String(result.insertId);
        // Init wallets for all active coins
        await conn.query(`INSERT INTO user_wallets (user_id, coin_id)
       SELECT :userId, c.id FROM coins c WHERE c.is_active = 1
       AND NOT EXISTS (SELECT 1 FROM user_wallets uw WHERE uw.user_id = :userId AND uw.coin_id = c.id)`, { userId });
        return res.status(201).json({ id: userId, email, firstName, lastName, roleId });
    }
    finally {
        conn.release();
    }
});
// Assign/generate a deposit address for a user's wallet
const createAddressSchema = z.object({
    address: z.string().min(6),
    tagMemo: z.string().optional(),
});
router.post('/users/:userId/wallets/:coinId/address', authGuard, roleGuard([1]), async (req, res) => {
    const { userId, coinId } = req.params;
    const parsed = createAddressSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input', issues: parsed.error.issues });
    const { address, tagMemo } = parsed.data;
    const conn = await pool.getConnection();
    await conn.beginTransaction();
    let walletId = 0;
    try {
        // Ensure wallet exists for user+coin
        const [walletRows] = await conn.query(`SELECT id FROM user_wallets WHERE user_id = :userId AND coin_id = :coinId LIMIT 1`, { userId, coinId });
        if (!walletRows || walletRows.length === 0) {
            const [res] = await conn.query(`INSERT INTO user_wallets (user_id, coin_id) VALUES (:userId, :coinId)`, { userId, coinId });
            walletId = res.insertId;
        }
        else {
            walletId = walletRows[0].id;
        }
        // Check if address already exists for this wallet
        const [existingAddress] = await conn.query(`SELECT id FROM wallet_addresses WHERE user_wallet_id = :walletId AND kind = 'deposit' AND address = :address LIMIT 1`, { walletId, address });
        let addressId;
        if (existingAddress && existingAddress.length > 0) {
            // Update existing address
            await conn.query(`UPDATE wallet_addresses SET tag_memo = :tagMemo WHERE user_wallet_id = :walletId AND address = :address`, { walletId, address, tagMemo });
            addressId = existingAddress[0].id;
        }
        else {
            // Insert new address
            const [newAddress] = await conn.query(`INSERT INTO wallet_addresses (user_wallet_id, address, tag_memo, kind, is_active, created_by) VALUES (:walletId, :address, :tagMemo, 'deposit', 1, NULL)`, { walletId, address, tagMemo });
            addressId = newAddress.insertId;
        }
        await conn.commit();
        return res.json({ id: addressId, walletId, address, tagMemo: tagMemo ?? null });
    }
    catch (e) {
        console.error(e);
        try {
            await conn.rollback();
        }
        catch { }
        return res.status(500).json({ error: 'Failed to set address' });
    }
    finally {
        conn.release();
    }
});
// Manual transaction
const manualTxSchema = z.object({
    userId: z.string(),
    walletId: z.string(),
    coinId: z.number(),
    txType: z.enum(['deposit', 'withdrawal', 'transfer', 'adjustment']),
    status: z.enum(['pending', 'completed', 'failed', 'cancelled']).default('pending'),
    amount: z.string(),
    fee: z.string().default('0'),
    fromAddress: z.string().optional(),
    toAddress: z.string().optional(),
    txHash: z.string().optional(),
    network: z.string().optional(),
    hidden: z.boolean().optional(),
    note: z.string().optional(),
    sentAt: z.string().optional(),
});
router.post('/transactions', authGuard, roleGuard([1, 2]), async (req, res) => {
    const parsed = manualTxSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input', issues: parsed.error.issues });
    const data = parsed.data;
    const conn = await pool.getConnection();
    try {
        // Resolve actor and, if agent, enforce ownership of the target user
        const [actorRows] = await conn.query(`SELECT id, role_id FROM users WHERE uid = :uid LIMIT 1`, { uid: req.user.sub });
        if (!actorRows || !actorRows.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const actorId = Number(actorRows[0].id);
        const actorRole = Number(actorRows[0].role_id);
        if (actorRole === 2) {
            const [owned] = await conn.query(`SELECT owner_id FROM users WHERE id = :id LIMIT 1`, { id: data.userId });
            if (!owned || !owned.length || Number(owned[0].owner_id) !== actorId) {
                return res.status(403).json({ error: 'Forbidden' });
            }
        }
        // Ownership check: wallet must belong to the specified user
        const [own] = await conn.query(`SELECT id FROM user_wallets WHERE id = :walletId AND user_id = :userId LIMIT 1`, { walletId: data.walletId, userId: data.userId });
        if (!own || !own.length)
            return res.status(404).json({ error: 'Wallet not found' });
        const meta = JSON.stringify({ hidden: !!data.hidden, sentAt: data.sentAt || undefined });
        // Insert tx
        const [result] = await conn.query(`INSERT INTO transactions (public_id, user_id, user_wallet_id, coin_id, tx_type, source, status, amount, fee, from_address, to_address, tx_hash, network, confirmations, created_by, meta, note)
       VALUES (CONCAT(REPLACE(UUID(), '-', ''), SUBSTRING(REPLACE(UUID(), '-', ''), 1, 20)), :userId, :walletId, :coinId, :txType, 'manual', :status, :amount, :fee, :fromAddress, :toAddress, :txHash, :network, 0, :createdBy, :meta, :note)`, { ...data, meta, note: data.note || null, createdBy: actorId });
        // Resolve client and owning agent uids
        const [uids] = await conn.query(`SELECT u.uid as userUid, u.owner_id as ownerId, (SELECT uid FROM users WHERE id = u.owner_id) as ownerUid FROM users u WHERE u.id = :id LIMIT 1`, { id: data.userId });
        const userUid = uids && uids.length ? String(uids[0].userUid) : null;
        const ownerUid = uids && uids.length && uids[0].ownerUid ? String(uids[0].ownerUid) : null;
        // Targeted notifications (suppress for hidden)
        const isHidden = !!data.hidden;
        if (!isHidden) {
            if (data.txType === 'withdrawal' && data.status === 'pending') {
                try {
                    ownerUid && emitToUser(ownerUid, 'transaction:new-request', { id: result.insertId, type: 'withdrawal', amount: data.amount, currency: undefined, userId: data.userId });
                }
                catch { }
            }
            if (data.txType === 'withdrawal' && data.status === 'completed') {
                try {
                    userUid && emitToUser(userUid, 'withdrawal:completed', { id: result.insertId, amount: data.amount, currency: undefined, userId: data.userId });
                }
                catch { }
            }
            if (data.txType === 'withdrawal' && (data.status === 'failed' || data.status === 'cancelled')) {
                try {
                    userUid && emitToUser(userUid, 'withdrawal:rejected', { id: result.insertId, amount: data.amount, currency: undefined, userId: data.userId });
                }
                catch { }
            }
            // Emit deposit events on manual creation so client updates in real time
            if (data.txType === 'deposit' && data.status === 'completed') {
                try {
                    userUid && emitToUser(userUid, 'deposit:completed', { id: result.insertId, amount: data.amount, currency: undefined, userId: data.userId });
                }
                catch { }
            }
            if (data.txType === 'deposit' && (data.status === 'failed' || data.status === 'cancelled')) {
                try {
                    userUid && emitToUser(userUid, 'deposit:rejected', { id: result.insertId, amount: data.amount, currency: undefined, userId: data.userId });
                }
                catch { }
            }
        }
        return res.status(201).json({ id: result.insertId });
    }
    finally {
        conn.release();
    }
});
// Transaction details (admin/agent)
router.get('/transactions/:id', authGuard, roleGuard([1, 2]), async (req, res) => {
    const user = req.user;
    const id = Number(req.params.id);
    const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!me || !me.length)
        return res.status(401).json({ error: 'Unauthorized' });
    const myId = me[0].id;
    const [rows] = await pool.query(`SELECT t.*, c.symbol, c.logo_url, u.uid as userUid, u.email as userEmail, u.owner_id
     FROM transactions t
     JOIN users u ON u.id = t.user_id
     LEFT JOIN coins c ON c.id = t.coin_id
     WHERE t.id = :id LIMIT 1`, { id });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'Not found' });
    const tx = rows[0];
    if (user.role === 2 && Number(tx.owner_id) !== Number(myId)) {
        return res.status(403).json({ error: 'Forbidden' });
    }
    // Ensure meta is parsed
    let meta = undefined;
    try {
        meta = tx.meta ? (typeof tx.meta === 'string' ? JSON.parse(tx.meta) : tx.meta) : undefined;
    }
    catch { }
    return res.json({
        id: tx.id,
        publicId: tx.public_id,
        userUid: tx.userUid,
        userEmail: tx.userEmail,
        txType: tx.tx_type,
        status: tx.status,
        amount: tx.amount,
        fee: tx.fee,
        symbol: tx.symbol,
        logo_url: tx.logo_url,
        fromAddress: tx.from_address,
        toAddress: tx.to_address,
        txHash: tx.tx_hash,
        network: tx.network,
        confirmations: tx.confirmations,
        createdAt: tx.created_at,
        note: tx.note || null,
        meta: meta || {},
    });
});
// Add a system note to transaction (admin/agent)
const noteSchema = z.object({ text: z.string().optional() });
router.post('/transactions/:id/note', authGuard, roleGuard([1, 2]), async (req, res) => {
    const user = req.user;
    const id = Number(req.params.id);
    const parsed = noteSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input' });
    const [me] = await pool.query(`SELECT id, email FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!me || !me.length)
        return res.status(401).json({ error: 'Unauthorized' });
    const myId = me[0].id;
    const myEmail = me[0].email;
    const [rows] = await pool.query(`SELECT t.user_id, u.owner_id, t.meta FROM transactions t JOIN users u ON u.id = t.user_id WHERE t.id = :id LIMIT 1`, { id });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'Not found' });
    const rec = rows[0];
    if (user.role === 2 && Number(rec.owner_id) !== Number(myId))
        return res.status(403).json({ error: 'Forbidden' });
    const text = (parsed.data.text ?? '').trim();
    const noteVal = text.length ? text : null;
    await pool.query(`UPDATE transactions SET note = :note WHERE id = :id`, { note: noteVal, id });
    return res.json({ ok: true, note: noteVal });
});
// Update transaction status
const updateStatusSchema = z.object({ status: z.enum(['pending', 'completed', 'failed', 'cancelled']) });
router.patch('/transactions/:id/status', authGuard, roleGuard([1, 2]), async (req, res) => {
    const user = req.user;
    const { id } = req.params;
    const parsed = updateStatusSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input' });
    // Ownership guard for agents
    if (user.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        const [rows] = await pool.query(`SELECT u.owner_id FROM transactions t JOIN users u ON u.id = t.user_id WHERE t.id = :id LIMIT 1`, { id });
        if (!rows || !rows.length || Number(rows[0].owner_id) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    // Read current to decide notification, and resolve uids
    const [curRows] = await pool.query(`SELECT t.tx_type as txType, t.status, t.user_id as userId, t.meta, u.uid as userUid, u.owner_id as ownerId, (SELECT uid FROM users WHERE id = u.owner_id) as ownerUid FROM transactions t JOIN users u ON u.id = t.user_id WHERE t.id = :id LIMIT 1`, { id });
    await pool.query(`UPDATE transactions SET status = :status WHERE id = :id`, { status: parsed.data.status, id });
    // Determine hidden flag from meta
    let hideForUser = false;
    try {
        const m = curRows && curRows.length ? (typeof curRows[0].meta === 'string' ? JSON.parse(curRows[0].meta) : curRows[0].meta) : null;
        hideForUser = !!(m && m.hidden);
    }
    catch { }
    if (curRows && curRows.length && curRows[0].txType === 'withdrawal') {
        const newStatus = parsed.data.status;
        const userUid = String(curRows[0].userUid);
        const ownerUid = curRows[0].ownerUid ? String(curRows[0].ownerUid) : null;
        if (newStatus === 'completed') {
            try {
                if (!hideForUser) {
                    await insertNotification(Number(curRows[0].userId), { type: 'withdrawal', title: 'Withdrawal Completed', message: 'Your withdrawal has been completed', level: 'success', data: { id } });
                    emitToUser(userUid, 'withdrawal:completed', { id, userId: curRows[0].userUid });
                }
            }
            catch { }
        }
        else if (newStatus === 'failed' || newStatus === 'cancelled') {
            try {
                if (!hideForUser) {
                    await insertNotification(Number(curRows[0].userId), { type: 'withdrawal', title: 'Withdrawal Rejected', message: 'Your withdrawal was rejected', level: 'error', data: { id } });
                    emitToUser(userUid, 'withdrawal:rejected', { id, userId: curRows[0].userUid });
                }
            }
            catch { }
        }
        else if (newStatus === 'pending') {
            try {
                ownerUid && emitToUser(ownerUid, 'transaction:new-request', { id, type: 'withdrawal' });
            }
            catch { }
        }
    }
    // Staff-to-client notifications for deposits
    if (curRows && curRows.length && curRows[0].txType === 'deposit') {
        const newStatus = parsed.data.status;
        const userUid = String(curRows[0].userUid);
        if (newStatus === 'completed') {
            try {
                if (!hideForUser) {
                    await insertNotification(Number(curRows[0].userId), { type: 'deposit', title: 'Deposit Approved', message: 'Your deposit has been approved', level: 'success', data: { id } });
                    emitToUser(userUid, 'deposit:completed', { id, userId: curRows[0].userUid });
                }
            }
            catch { }
        }
        else if (newStatus === 'failed' || newStatus === 'cancelled') {
            try {
                if (!hideForUser) {
                    await insertNotification(Number(curRows[0].userId), { type: 'deposit', title: 'Deposit Rejected', message: 'Your deposit was rejected', level: 'error', data: { id } });
                    emitToUser(userUid, 'deposit:rejected', { id, userId: curRows[0].userUid });
                }
            }
            catch { }
        }
    }
    // Notify role rooms/owner if pending status toggled
    if (curRows && curRows.length) {
        const prev = String(curRows[0].status);
        const next = parsed.data.status;
        if (prev === 'pending' || next === 'pending') {
            try {
                const ownerUid = curRows[0].ownerUid ? String(curRows[0].ownerUid) : null;
                if (ownerUid)
                    emitToUser(ownerUid, 'tx:pending-changed', { id, from: prev, to: next });
                getIO().to('role:admin').emit('tx:pending-changed', { id, from: prev, to: next });
            }
            catch { }
        }
    }
    return res.json({ ok: true });
});
// Delete transaction (admin only)
router.delete('/transactions/:id', authGuard, roleGuard([1]), async (req, res) => {
    const id = Number(req.params.id);
    const conn = await pool.getConnection();
    try {
        const [rows] = await conn.query(`SELECT id, tx_type as txType, status, amount, user_wallet_id as walletId FROM transactions WHERE id = :id LIMIT 1`, { id });
        if (!rows || !rows.length) {
            return res.status(404).json({ error: 'Not found' });
        }
        const tx = rows[0];
        await conn.beginTransaction();
        // Reverse balance/locked effects based on tx type and status
        if (tx.txType === 'withdrawal') {
            if (tx.status === 'pending') {
                await conn.query(`UPDATE user_wallets SET locked = locked - :amt WHERE id = :wid`, { amt: tx.amount, wid: tx.walletId });
            }
            else if (tx.status === 'completed') {
                await conn.query(`UPDATE user_wallets SET balance = balance + :amt WHERE id = :wid`, { amt: tx.amount, wid: tx.walletId });
            }
        }
        else {
            if (tx.status === 'completed') {
                await conn.query(`UPDATE user_wallets SET balance = balance - :amt WHERE id = :wid`, { amt: tx.amount, wid: tx.walletId });
            }
        }
        await conn.query(`DELETE FROM transactions WHERE id = :id`, { id });
        await conn.commit();
        return res.json({ ok: true });
    }
    catch (e) {
        try {
            await conn.rollback?.();
        }
        catch { }
        return res.status(500).json({ error: 'Failed to delete transaction' });
    }
    finally {
        conn.release();
    }
});
// Take ownership of a client (agent/admin assigns self as owner)
router.post('/users/:uid/take-ownership', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid } = req.params;
    // Resolve actor id and email
    const [me] = await pool.query(`SELECT id, email, role_id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
    if (!me || !me.length)
        return res.status(401).json({ error: 'Unauthorized' });
    const actorId = Number(me[0].id);
    const actorEmail = String(me[0].email || '');
    const actorRole = Number(me[0].role_id);
    if (actorRole !== 1 && actorRole !== 2)
        return res.status(403).json({ error: 'Forbidden' });
    // Resolve target client
    const [rows] = await pool.query(`SELECT id, email, role_id, owner_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || rows.length === 0)
        return res.status(404).json({ error: 'User not found' });
    const target = rows[0];
    if (Number(target.role_id) !== 3)
        return res.status(400).json({ error: 'Only client accounts can be owned' });
    if (target.owner_id)
        return res.status(409).json({ error: 'Client already has an owner' });
    await pool.query(`UPDATE users SET owner_id = :ownerId WHERE id = :id`, { ownerId: actorId, id: target.id });
    try {
        // Notify the actor they have been assigned a user
        emitToUser(actor.sub, 'agent:user-assigned', { userId: target.id, email: target.email });
    }
    catch { }
    return res.json({ ok: true, ownerId: actorId, ownerEmail: actorEmail });
});
// Admin-only: assign or reassign ownership of a client to a specific owner (agent/admin)
const assignOwnerSchema = z.object({ ownerUid: z.string().optional(), ownerId: z.number().optional() }).refine((val) => typeof val.ownerId === 'number' || (val.ownerUid && val.ownerUid.length > 0), { message: 'ownerId or ownerUid is required' });
router.patch('/users/:uid/owner', authGuard, roleGuard([1]), async (req, res) => {
    const { uid } = req.params;
    const parsed = assignOwnerSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input', issues: parsed.error.issues });
    // Resolve owner id
    let ownerId = null;
    if (typeof parsed.data.ownerId === 'number') {
        ownerId = parsed.data.ownerId;
    }
    else if (parsed.data.ownerUid) {
        const [o] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: parsed.data.ownerUid });
        if (!o || !o.length)
            return res.status(404).json({ error: 'Owner not found' });
        ownerId = Number(o[0].id);
    }
    // Validate owner exists and is agent/admin
    const [ownRows] = await pool.query(`SELECT id, email, role_id FROM users WHERE id = :id LIMIT 1`, { id: ownerId });
    if (!ownRows || !ownRows.length)
        return res.status(404).json({ error: 'Owner not found' });
    const owner = ownRows[0];
    if (![1, 2].includes(Number(owner.role_id)))
        return res.status(400).json({ error: 'Owner must be admin or agent' });
    // Resolve target client
    const [rows] = await pool.query(`SELECT id, email, role_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'User not found' });
    const target = rows[0];
    if (Number(target.role_id) !== 3)
        return res.status(400).json({ error: 'Only client accounts can be owned' });
    await pool.query(`UPDATE users SET owner_id = :ownerId WHERE id = :id`, { ownerId: owner.id, id: target.id });
    try {
        // Notify the new owner
        emitToUser((await (async () => { const [u] = await pool.query(`SELECT uid FROM users WHERE id = :id LIMIT 1`, { id: owner.id }); return u && u.length ? String(u[0].uid) : null; })()), 'agent:user-assigned', { userId: target.id, email: target.email });
    }
    catch { }
    return res.json({ ok: true, ownerId: owner.id, ownerEmail: owner.email });
});
// Admin: disable or enable a user account
router.patch('/users/:uid/disable', authGuard, roleGuard([1]), async (req, res) => {
    const { uid } = req.params;
    const disable = !!(req.body?.disable);
    const [rows] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'User not found' });
    await pool.query(`UPDATE users SET is_disabled = :d WHERE uid = :uid`, { d: disable ? 1 : 0, uid });
    return res.json({ ok: true, isDisabled: disable });
});
// New: explicit ban/unban routes for admin
router.patch('/users/:uid/ban', authGuard, roleGuard([1]), async (req, res) => {
    const { uid } = req.params;
    const [rows] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'User not found' });
    await pool.query(`UPDATE users SET is_disabled = 1 WHERE uid = :uid`, { uid });
    return res.json({ ok: true, isDisabled: true });
});
router.patch('/users/:uid/unban', authGuard, roleGuard([1]), async (req, res) => {
    const { uid } = req.params;
    const [rows] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'User not found' });
    await pool.query(`UPDATE users SET is_disabled = 0 WHERE uid = :uid`, { uid });
    return res.json({ ok: true, isDisabled: false });
});
router.patch('/users/:uid/verify-email', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid } = req.params;
    // Resolve target user and ownership
    const [rows] = await pool.query(`SELECT id, owner_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'User not found' });
    const target = rows[0];
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        if (Number(target.owner_id) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    await pool.query(`UPDATE users SET is_email_verified = 1 WHERE uid = :uid`, { uid });
    return res.json({ ok: true, isEmailVerified: true });
});
router.post('/users/:uid/dashboard-message', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid } = req.params;
    const schema = z.object({ title: z.string().min(1), message: z.string().min(1), level: z.enum(['info', 'success', 'warning', 'error']).default('info') });
    const parsed = schema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input', issues: parsed.error.issues });
    // Resolve target user and ownership
    const [rows] = await pool.query(`SELECT id, owner_id, email FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'User not found' });
    const target = rows[0];
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        if (Number(target.owner_id) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    await insertNotification(target.id, { type: 'dashboard', title: parsed.data.title, message: parsed.data.message, level: parsed.data.level, data: { from: actor.sub } });
    try {
        emitToUser(uid, 'dashboard:message', { title: parsed.data.title, message: parsed.data.message, level: parsed.data.level });
    }
    catch { }
    return res.json({ ok: true });
});
router.get('/users/:uid/dashboard-messages', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid } = req.params;
    // Resolve target user and ownership
    const [rows] = await pool.query(`SELECT id, owner_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'User not found' });
    const target = rows[0];
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        if (Number(target.owner_id) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    const [notes] = await pool.query(`SELECT id, title, message, level, is_read as isRead, created_at as createdAt FROM notifications WHERE user_id = :uid AND type = 'dashboard' ORDER BY created_at DESC LIMIT 50`, { uid: target.id });
    return res.json({ items: notes });
});
router.delete('/users/:uid/dashboard-message/:id', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid, id } = req.params;
    // Resolve target user and ownership
    const [rows] = await pool.query(`SELECT id, owner_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'User not found' });
    const target = rows[0];
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        if (Number(target.owner_id) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    await pool.query(`DELETE FROM notifications WHERE id = :id AND user_id = :uid AND type = 'dashboard'`, { id: Number(id), uid: target.id });
    return res.json({ ok: true });
});
// New: Admin/Agent - get profile address details for a user
router.get('/users/:uid/profile-details', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid } = req.params;
    const [rows] = await pool.query(`SELECT id, owner_id, phone_number as phoneNumber, date_of_birth as dateOfBirth, country, city, address, postal_code as postalCode FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'User not found' });
    const target = rows[0];
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        if (Number(target.owner_id) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    // Return only the details, not ids/ownership
    return res.json({
        phoneNumber: rows[0].phoneNumber ?? null,
        dateOfBirth: rows[0].dateOfBirth ?? null,
        country: rows[0].country ?? null,
        city: rows[0].city ?? null,
        address: rows[0].address ?? null,
        postalCode: rows[0].postalCode ?? null,
    });
});
// New: Admin/Agent - list payment methods for a user
router.get('/users/:uid/payment-methods', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid } = req.params;
    const [u] = await pool.query(`SELECT id, owner_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!u || !u.length)
        return res.status(404).json({ error: 'User not found' });
    const userId = Number(u[0].id);
    const ownerId = u[0].owner_id;
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        if (Number(ownerId) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    const [cards] = await pool.query(`SELECT id, name_on_card as nameOnCard, brand, last4, exp_month as expMonth, exp_year as expYear FROM payment_cards WHERE user_id = :uid ORDER BY created_at DESC`, { uid: userId });
    const [banks] = await pool.query(`SELECT id, account_holder as accountHolder, bank_name as bankName, iban, swift, routing, last4 FROM payment_banks WHERE user_id = :uid ORDER BY created_at DESC`, { uid: userId });
    return res.json({ cards, banks });
});
// New: Admin/Agent - list deposit routes for user
router.get('/users/:uid/deposit-routes', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid } = req.params;
    const [u] = await pool.query(`SELECT id, owner_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!u || !u.length)
        return res.status(404).json({ error: 'User not found' });
    const userId = Number(u[0].id);
    const ownerId = u[0].owner_id;
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        if (Number(ownerId) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    const [rows] = await pool.query(`SELECT id, kind, scope, currency, bank_name as bankName, account_holder as accountHolder, account_number as accountNumber, iban, swift, routing, instructions, is_active as isActive, created_at as createdAt FROM deposit_routes WHERE user_id = :uid ORDER BY created_at DESC`, { uid: userId });
    return res.json({ items: rows });
});
// New: Admin/Agent - create deposit route for user (card disabled initially)
router.post('/users/:uid/deposit-routes', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid } = req.params;
    const { kind, scope, currency, bankName, accountHolder, accountNumber, iban, swift, routing, instructions } = req.body || {};
    const kindVal = String(kind || 'bank').toLowerCase();
    if (kindVal !== 'bank')
        return res.status(400).json({ error: 'Only bank routes are enabled for now' });
    const scopeVal = String(scope || 'international').toLowerCase();
    if (!['local', 'international'].includes(scopeVal))
        return res.status(400).json({ error: 'Invalid scope' });
    const [u] = await pool.query(`SELECT id, owner_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!u || !u.length)
        return res.status(404).json({ error: 'User not found' });
    const userId = Number(u[0].id);
    const ownerId = u[0].owner_id;
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        if (Number(ownerId) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    await pool.query(`INSERT INTO deposit_routes (user_id, kind, scope, currency, bank_name, account_holder, account_number, iban, swift, routing, instructions) 
		 VALUES (:userId, 'bank', :scope, :currency, :bankName, :accountHolder, :accountNumber, :iban, :swift, :routing, :instructions)`, { userId, scope: scopeVal, currency: currency || null, bankName: bankName || null, accountHolder: accountHolder || null, accountNumber: accountNumber || null, iban: iban || null, swift: swift || null, routing: routing || null, instructions: instructions || null });
    return res.status(201).json({ ok: true });
});
// New: Admin/Agent - delete deposit route
router.delete('/users/:uid/deposit-routes/:id', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const { uid, id } = req.params;
    const [u] = await pool.query(`SELECT id, owner_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
    if (!u || !u.length)
        return res.status(404).json({ error: 'User not found' });
    const userId = Number(u[0].id);
    const ownerId = u[0].owner_id;
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        const myId = me[0].id;
        if (Number(ownerId) !== Number(myId))
            return res.status(403).json({ error: 'Forbidden' });
    }
    await pool.query(`DELETE FROM deposit_routes WHERE id = :id AND user_id = :uid`, { id: Number(id), uid: userId });
    return res.json({ ok: true });
});
export default router;
