import { Router } from 'express';
import { z } from 'zod';
import bcrypt from 'bcryptjs';
import pool from '../config/db.js';
import { cookieOptions, signJwt } from '../utils/jwt.js';
import { authGuard } from '../middleware/auth.js';
import crypto from 'crypto';
import { getIO, emitToUser } from '../socket.js';
const ALLOWED_TIMEZONES = new Set([
    'Europe/London', 'Europe/Paris', 'Europe/Berlin', 'Europe/Madrid', 'Europe/Rome', 'UTC',
    'America/New_York', 'America/Chicago', 'America/Denver', 'America/Los_Angeles', 'America/Sao_Paulo',
    'Asia/Dubai', 'Asia/Kolkata', 'Asia/Singapore', 'Asia/Tokyo', 'Australia/Sydney'
]);
const router = Router();
const registerSchema = z.object({
    firstName: z.string().min(1),
    lastName: z.string().min(1),
    email: z.string().email(),
    password: z.string().min(8),
    referralCode: z.string().optional(),
});
const loginSchema = z.object({
    email: z.string().email(),
    password: z.string().min(1),
});
router.post('/register', async (req, res) => {
    const parsed = registerSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input', issues: parsed.error.issues });
    const { firstName, lastName, email, password, referralCode } = parsed.data;
    const conn = await pool.getConnection();
    try {
        const [existing] = await conn.query('SELECT id FROM users WHERE email = :email LIMIT 1', { email });
        if (Array.isArray(existing) && existing.length > 0)
            return res.status(409).json({ error: 'Email already registered' });
        const hash = await bcrypt.hash(password, 10);
        const roleId = 3; // client
        let ownerId = null;
        if (referralCode) {
            const [rows] = await conn.query(`SELECT owner_user_id FROM referral_codes WHERE code = :code LIMIT 1`, { code: referralCode });
            if (rows && rows.length)
                ownerId = rows[0].owner_user_id;
        }
        const [result] = await conn.query('INSERT INTO users (role_id, owner_id, email, password_hash, first_name, last_name, uid) VALUES (:roleId, :ownerId, :email, :hash, :firstName, :lastName, UUID())', {
            roleId,
            ownerId,
            email,
            hash,
            firstName,
            lastName,
        });
        const userId = String(result.insertId);
        // Fetch generated uid
        const [urows] = await conn.query(`SELECT uid FROM users WHERE id = :id`, { id: userId });
        const uid = urows[0].uid;
        // Initialize wallets for active coins
        await conn.query(`INSERT INTO user_wallets (user_id, coin_id) 
       SELECT :userId, c.id FROM coins c WHERE c.is_active = 1 
       AND NOT EXISTS (SELECT 1 FROM user_wallets uw WHERE uw.user_id = :userId AND uw.coin_id = c.id)`, { userId });
        // Notify admins: keep broad event if your UI listens for admin boards; otherwise, remove
        try {
            getIO().emit('admin:new-client', { id: uid, email, firstName, lastName, ownerId });
        }
        catch { }
        // Notify owning agent only
        if (ownerId != null) {
            const [agentRows] = await conn.query(`SELECT uid FROM users WHERE id = :id LIMIT 1`, { id: ownerId });
            const agentUid = agentRows && agentRows.length ? String(agentRows[0].uid) : null;
            if (agentUid) {
                try {
                    emitToUser(agentUid, 'agent:user-assigned', { userId: uid, email, ownerId });
                }
                catch { }
            }
        }
        // Send email verification link
        try {
            const verifyToken = crypto.randomBytes(32).toString('hex');
            const verifyExpiresAt = new Date(Date.now() + 1000 * 60 * 60); // 60 minutes
            await conn.query('INSERT INTO email_verification_tokens (user_id, token, expires_at) VALUES (:userId, :token, :expiresAt)', { userId, token: verifyToken, expiresAt: verifyExpiresAt });
            const { getEmailService } = await import('../services/email/index.js');
            await getEmailService().sendVerifyEmail(email, verifyToken, { userName: firstName || 'there', expiresMinutes: 60 });
        }
        catch { }
        const token = signJwt({ sub: uid, role: roleId });
        res.cookie('token', token, cookieOptions());
        return res.status(201).json({ id: uid, email, firstName, lastName, role: roleId });
    }
    finally {
        conn.release();
    }
});
router.post('/login', async (req, res) => {
    const parsed = loginSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input' });
    const { email, password } = parsed.data;
    const [rows] = await pool.query('SELECT id, uid, role_id, is_disabled, password_hash, first_name, last_name FROM users WHERE email = :email LIMIT 1', { email });
    if (!rows || rows.length === 0)
        return res.status(401).json({ error: 'Invalid credentials' });
    const u = rows[0];
    const ok = await bcrypt.compare(password, u.password_hash);
    if (!ok)
        return res.status(401).json({ error: 'Invalid credentials' });
    if (u.is_disabled)
        return res.status(403).json({ error: 'Account is banned' });
    const ip = String(req.headers['x-forwarded-for'] || req.socket.remoteAddress || '');
    const ua = String(req.headers['user-agent'] || '');
    await pool.query('UPDATE users SET last_login_at = NOW(), last_login_ip = :ip, last_login_ua = :ua WHERE id = :id', { id: u.id, ip, ua });
    const token = signJwt({ sub: String(u.uid), role: u.role_id });
    res.cookie('token', token, cookieOptions());
    return res.json({ id: u.uid, email, firstName: u.first_name, lastName: u.last_name, role: u.role_id });
});
router.post('/logout', (_req, res) => {
    res.clearCookie('token', { path: '/' });
    res.json({ ok: true });
});
router.get('/me', authGuard, async (req, res) => {
    const user = req.user;
    const [rows] = await pool.query('SELECT id, uid, role_id, email, first_name, last_name, is_verified, is_email_verified, preferences, phone_number, date_of_birth, country, city, address, postal_code FROM users WHERE uid = :uid LIMIT 1', { uid: user.sub });
    if (!rows || rows.length === 0)
        return res.status(404).json({ error: 'Not found' });
    const u = rows[0];
    let prefs = undefined;
    if (u.preferences != null) {
        try {
            prefs = typeof u.preferences === 'string' ? JSON.parse(u.preferences) : u.preferences;
        }
        catch {
            prefs = undefined;
        }
    }
    if (!prefs || typeof prefs !== 'object')
        prefs = {};
    if (!prefs.timezone || typeof prefs.timezone !== 'string' || !prefs.timezone.trim()) {
        prefs.timezone = 'Europe/London';
    }
    res.json({
        id: u.id,
        uid: u.uid,
        role: u.role_id,
        email: u.email,
        firstName: u.first_name,
        lastName: u.last_name,
        isVerified: !!u.is_verified,
        isEmailVerified: !!u.is_email_verified,
        preferences: prefs,
        phoneNumber: u.phone_number ?? null,
        dateOfBirth: u.date_of_birth ?? null,
        country: u.country ?? null,
        city: u.city ?? null,
        address: u.address ?? null,
        postalCode: u.postal_code ?? null,
    });
});
// Add: Update profile (email is immutable)
const profileSchema = z.object({
    firstName: z.string().min(1).optional(),
    lastName: z.string().min(1).optional(),
    phoneNumber: z.string().min(1).optional(),
    dateOfBirth: z.string().min(4).optional(),
    country: z.string().min(1).optional(),
    city: z.string().min(1).optional(),
    address: z.string().min(1).optional(),
    postalCode: z.string().min(1).optional(),
}).strict().partial();
router.post('/profile', authGuard, async (req, res) => {
    const user = req.user;
    const parsed = profileSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input', issues: parsed.error.issues });
    const data = parsed.data;
    if (!Object.keys(data).length)
        return res.json({ ok: true });
    const [u] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!u || !u.length)
        return res.status(404).json({ error: 'Not found' });
    const userId = u[0].id;
    const fields = [];
    const params = { id: userId };
    if (typeof data.firstName === 'string') {
        fields.push('first_name = :firstName');
        params.firstName = data.firstName;
    }
    if (typeof data.lastName === 'string') {
        fields.push('last_name = :lastName');
        params.lastName = data.lastName;
    }
    if (typeof data.phoneNumber === 'string') {
        fields.push('phone_number = :phoneNumber');
        params.phoneNumber = data.phoneNumber;
    }
    if (typeof data.dateOfBirth === 'string') {
        fields.push('date_of_birth = :dateOfBirth');
        params.dateOfBirth = data.dateOfBirth;
    }
    if (typeof data.country === 'string') {
        fields.push('country = :country');
        params.country = data.country;
    }
    if (typeof data.city === 'string') {
        fields.push('city = :city');
        params.city = data.city;
    }
    if (typeof data.address === 'string') {
        fields.push('address = :address');
        params.address = data.address;
    }
    if (typeof data.postalCode === 'string') {
        fields.push('postal_code = :postalCode');
        params.postalCode = data.postalCode;
    }
    if (fields.length) {
        await pool.query(`UPDATE users SET ${fields.join(', ')} WHERE id = :id`, params);
    }
    return res.json({ ok: true });
});
// Add: Change password for logged-in user
const changePasswordSchema = z.object({
    currentPassword: z.string().min(1),
    newPassword: z.string().min(8),
});
router.post('/change-password', authGuard, async (req, res) => {
    const user = req.user;
    const parsed = changePasswordSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input', issues: parsed.error.issues });
    const { currentPassword, newPassword } = parsed.data;
    const [rows] = await pool.query('SELECT id, password_hash FROM users WHERE uid = :uid LIMIT 1', { uid: user.sub });
    if (!rows || rows.length === 0)
        return res.status(404).json({ error: 'Not found' });
    const rec = rows[0];
    const ok = await bcrypt.compare(currentPassword, rec.password_hash);
    if (!ok)
        return res.status(401).json({ error: 'Current password is incorrect' });
    const hash = await bcrypt.hash(newPassword, 10);
    await pool.query('UPDATE users SET password_hash = :hash WHERE id = :id', { hash, id: rec.id });
    return res.json({ ok: true });
});
const preferencesSchema = z.object({
    theme: z.enum(['light', 'dark', 'auto']).optional(),
    language: z.string().optional(),
    currency: z.string().optional(),
    timezone: z.string().optional(),
    notifications: z.any().optional(),
    privacy: z.any().optional(),
});
router.post('/preferences', authGuard, async (req, res) => {
    const user = req.user;
    const parsed = preferencesSchema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid preferences' });
    const [u] = await pool.query(`SELECT id, preferences FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!u || !u.length)
        return res.status(404).json({ error: 'Not found' });
    const userId = u[0].id;
    let current = {};
    if (u[0].preferences != null) {
        try {
            current = typeof u[0].preferences === 'string' ? JSON.parse(u[0].preferences) : u[0].preferences;
        }
        catch {
            current = {};
        }
    }
    const next = { ...current, ...parsed.data };
    // Validate timezone and default to Europe/London if invalid
    if (next.timezone && typeof next.timezone === 'string') {
        if (!ALLOWED_TIMEZONES.has(next.timezone)) {
            return res.status(400).json({ error: 'Invalid timezone' });
        }
    }
    if (!next.timezone)
        next.timezone = 'Europe/London';
    await pool.query('UPDATE users SET preferences = :prefs WHERE id = :id', { prefs: JSON.stringify(next), id: userId });
    return res.json({ ok: true, preferences: next });
});
// Forgot Password - issue token (in real app, email it to user)
router.post('/forgot-password', async (req, res) => {
    const schema = z.object({ email: z.string().email() });
    const parsed = schema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid email' });
    const { email } = parsed.data;
    const conn = await pool.getConnection();
    try {
        const [rows] = await conn.query('SELECT id FROM users WHERE email = :email LIMIT 1', { email });
        if (!rows || rows.length === 0)
            return res.json({ ok: true }); // don't reveal existence
        const userId = rows[0].id;
        const token = crypto.randomBytes(32).toString('hex');
        const expiresAt = new Date(Date.now() + 1000 * 60 * 30); // 30 minutes
        await conn.query('INSERT INTO password_reset_tokens (user_id, token, expires_at) VALUES (:userId, :token, :expiresAt)', { userId, token, expiresAt });
        try {
            const [u] = await conn.query(`SELECT first_name, last_name FROM users WHERE id = :id LIMIT 1`, { id: userId });
            const firstName = (u && u.length ? u[0].first_name : null);
            const { getEmailService } = await import('../services/email/index.js');
            await getEmailService().sendPasswordReset(email, token, { userName: firstName || 'there', expiresMinutes: 30 });
        }
        catch { }
        // For dev, also return token to help testing
        return res.json({ ok: true, token });
    }
    finally {
        conn.release();
    }
});
// Reset Password - verify token and update password
router.post('/reset-password', async (req, res) => {
    const schema = z.object({ token: z.string().min(10), password: z.string().min(8) });
    const parsed = schema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input' });
    const { token, password } = parsed.data;
    const conn = await pool.getConnection();
    try {
        const [rows] = await conn.query('SELECT pr.user_id, pr.expires_at, pr.used FROM password_reset_tokens pr WHERE pr.token = :token LIMIT 1', { token });
        if (!rows || rows.length === 0)
            return res.status(400).json({ error: 'Invalid token' });
        const rec = rows[0];
        if (rec.used)
            return res.status(400).json({ error: 'Token already used' });
        if (new Date(rec.expires_at).getTime() < Date.now())
            return res.status(400).json({ error: 'Token expired' });
        const hash = await bcrypt.hash(password, 10);
        await conn.beginTransaction();
        await conn.query('UPDATE users SET password_hash = :hash WHERE id = :id', { hash, id: rec.user_id });
        await conn.query('UPDATE password_reset_tokens SET used = 1 WHERE token = :token', { token });
        await conn.commit();
        return res.json({ ok: true });
    }
    catch (e) {
        await pool.rollback?.();
        throw e;
    }
    finally {
        conn.release();
    }
});
// Send email verification link for the current user
router.post('/verify-email/send', authGuard, async (req, res) => {
    const user = req.user;
    const conn = await pool.getConnection();
    try {
        const [rows] = await conn.query(`SELECT id, email, first_name FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
        if (!rows || !rows.length)
            return res.status(404).json({ error: 'Not found' });
        const u = rows[0];
        const token = crypto.randomBytes(32).toString('hex');
        const expiresAt = new Date(Date.now() + 1000 * 60 * 60); // 60 minutes
        await conn.query('INSERT INTO email_verification_tokens (user_id, token, expires_at) VALUES (:userId, :token, :expiresAt)', { userId: u.id, token, expiresAt });
        try {
            const { getEmailService } = await import('../services/email/index.js');
            await getEmailService().sendVerifyEmail(u.email, token, { userName: u.first_name || 'there', expiresMinutes: 60 });
        }
        catch { }
        return res.json({ ok: true });
    }
    finally {
        conn.release();
    }
});
// Confirm email verification
router.post('/verify-email', async (req, res) => {
    const schema = z.object({ token: z.string().min(10) });
    const parsed = schema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input' });
    const { token } = parsed.data;
    const conn = await pool.getConnection();
    try {
        const [rows] = await conn.query('SELECT user_id, expires_at, used FROM email_verification_tokens WHERE token = :token LIMIT 1', { token });
        if (!rows || !rows.length)
            return res.status(400).json({ error: 'Invalid token' });
        const rec = rows[0];
        if (rec.used)
            return res.status(400).json({ error: 'Token already used' });
        if (new Date(rec.expires_at).getTime() < Date.now())
            return res.status(400).json({ error: 'Token expired' });
        await conn.beginTransaction();
        await conn.query('UPDATE users SET is_email_verified = 1 WHERE id = :id', { id: rec.user_id });
        await conn.query('UPDATE email_verification_tokens SET used = 1 WHERE token = :token', { token });
        await conn.commit();
        return res.json({ ok: true });
    }
    catch (e) {
        try {
            await conn.rollback?.();
        }
        catch { }
        throw e;
    }
    finally {
        conn.release();
    }
});
// Request a 2FA code via email (login purpose)
router.post('/2fa/email/request', async (req, res) => {
    const schema = z.object({ email: z.string().email() });
    const parsed = schema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid email' });
    const { email } = parsed.data;
    const conn = await pool.getConnection();
    try {
        const [rows] = await conn.query(`SELECT id, first_name FROM users WHERE email = :email LIMIT 1`, { email });
        if (!rows || !rows.length)
            return res.json({ ok: true }); // do not reveal existence
        const u = rows[0];
        const code = String(crypto.randomInt(0, 1000000)).padStart(6, '0');
        const expiresAt = new Date(Date.now() + 1000 * 60 * 10); // 10 minutes
        await conn.query(`INSERT INTO email_otp_codes (user_id, code, purpose, expires_at) VALUES (:userId, :code, 'login', :expiresAt)`, { userId: u.id, code, expiresAt });
        try {
            const { getEmailService } = await import('../services/email/index.js');
            await getEmailService().sendTwoStepCode(email, code, { userName: u.first_name || 'there', expiresMinutes: 10 });
        }
        catch { }
        return res.json({ ok: true });
    }
    finally {
        conn.release();
    }
});
// Verify a 2FA email code (login purpose)
router.post('/2fa/email/verify', async (req, res) => {
    const schema = z.object({ email: z.string().email(), code: z.string().length(6) });
    const parsed = schema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input' });
    const { email, code } = parsed.data;
    const conn = await pool.getConnection();
    try {
        const [rows] = await conn.query(`SELECT eoc.id, eoc.user_id, eoc.expires_at, eoc.used
       FROM email_otp_codes eoc
       JOIN users u ON u.id = eoc.user_id
       WHERE u.email = :email AND eoc.code = :code AND eoc.purpose = 'login'
       ORDER BY eoc.id DESC
       LIMIT 1`, { email, code });
        if (!rows || !rows.length)
            return res.status(400).json({ error: 'Invalid code' });
        const rec = rows[0];
        if (rec.used)
            return res.status(400).json({ error: 'Code already used' });
        if (new Date(rec.expires_at).getTime() < Date.now())
            return res.status(400).json({ error: 'Code expired' });
        await conn.query(`UPDATE email_otp_codes SET used = 1 WHERE id = :id`, { id: rec.id });
        return res.json({ ok: true });
    }
    finally {
        conn.release();
    }
});
export default router;
