import { Router } from 'express';
import pool from '../config/db.js';
import { authGuard } from '../middleware/auth.js';
import { roleGuard } from '../middleware/roles.js';
import { emitToUser } from '../socket.js';
const router = Router();
// Unified orders endpoints
router.post('/orders', authGuard, async (req, res) => {
    const user = req.user;
    const { type, coinId, coinSymbol, amountCoin, amountFiat, currency, depositRouteId, payoutCardId, payoutBankId } = req.body || {};
    if (!type || !['buy', 'sell'].includes(String(type)))
        return res.status(400).json({ error: 'Invalid type' });
    if (!coinId || !coinSymbol || !amountCoin || !amountFiat || !currency)
        return res.status(400).json({ error: 'Missing fields' });
    const [me] = await pool.query(`SELECT id, is_email_verified, is_verified, owner_id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!me || !me.length)
        return res.status(401).json({ error: 'Unauthorized' });
    if (!me[0].is_email_verified || !me[0].is_verified)
        return res.status(400).json({ error: 'Account not verified' });
    const userId = Number(me[0].id);
    if (String(type) === 'sell') {
        const conn = await pool.getConnection();
        try {
            await conn.beginTransaction();
            // Create order first to get id
            const [ins] = await conn.query(`INSERT INTO orders (user_id, type, coin_id, coin_symbol, amount_coin, amount_fiat, currency, deposit_route_id, payout_card_id, payout_bank_id)
				 VALUES (:userId, :type, :coinId, :coinSymbol, :amountCoin, :amountFiat, :currency, :depositRouteId, :payoutCardId, :payoutBankId)`, { userId, type: 'sell', coinId: Number(coinId), coinSymbol: String(coinSymbol).toUpperCase(), amountCoin: String(amountCoin), amountFiat: String(amountFiat), currency: String(currency).toUpperCase(), depositRouteId: null, payoutCardId: payoutCardId ? Number(payoutCardId) : null, payoutBankId: payoutBankId ? Number(payoutBankId) : null });
            const orderId = Number(ins.insertId);
            // Ensure wallet exists and has available balance; lock amount by increasing locked if available
            const [walletRows] = await conn.query(`SELECT id FROM user_wallets WHERE user_id = :uid AND coin_id = :coinId LIMIT 1`, { uid: userId, coinId: Number(coinId) });
            if (!walletRows || !walletRows.length) {
                await conn.rollback();
                return res.status(400).json({ error: 'No wallet for this asset' });
            }
            const walletId = Number(walletRows[0].id);
            const [upd] = await conn.query(`UPDATE user_wallets SET locked = locked + :amt WHERE id = :walletId AND (balance - locked) >= :amt`, { amt: String(amountCoin), walletId });
            if (!upd || Number(upd.affectedRows || 0) === 0) {
                await conn.rollback();
                return res.status(400).json({ error: 'Insufficient available balance' });
            }
            await conn.commit();
            return res.status(201).json({ ok: true, id: orderId });
        }
        catch (e) {
            try {
                await conn.rollback?.();
            }
            catch { }
            return res.status(500).json({ error: 'Failed to create order' });
        }
        finally {
            // @ts-ignore
            conn.release();
        }
    }
    else {
        await pool.query(`INSERT INTO orders (user_id, type, coin_id, coin_symbol, amount_coin, amount_fiat, currency, deposit_route_id, payout_card_id, payout_bank_id)
			 VALUES (:userId, :type, :coinId, :coinSymbol, :amountCoin, :amountFiat, :currency, :depositRouteId, :payoutCardId, :payoutBankId)`, { userId, type: 'buy', coinId: Number(coinId), coinSymbol: String(coinSymbol).toUpperCase(), amountCoin: String(amountCoin), amountFiat: String(amountFiat), currency: String(currency).toUpperCase(), depositRouteId: depositRouteId ? Number(depositRouteId) : null, payoutCardId: null, payoutBankId: null });
        return res.status(201).json({ ok: true });
    }
});
router.get('/my-orders', authGuard, async (req, res) => {
    const user = req.user;
    const status = String(req.query.status || '').trim();
    const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!me || !me.length)
        return res.status(401).json({ error: 'Unauthorized' });
    const params = { userId: Number(me[0].id) };
    let where = 'o.user_id = :userId';
    if (status) {
        params.status = status;
        where += ' AND o.status = :status';
    }
    const [rows] = await pool.query(`SELECT o.id, o.type, o.coin_id as coinId, o.coin_symbol as coinSymbol, o.amount_coin as amountCoin, o.amount_fiat as amountFiat, o.currency, o.status, o.reject_reason as rejectReason, o.created_at as createdAt,
		 c.name as coinName, c.logo_url as coinLogo,
		 dr.bank_name as depositBankName, dr.account_holder as depositAccountHolder, dr.iban as depositIban,
		 pc.name_on_card as payoutCardName, pc.last4 as payoutCardLast4,
		 pb.bank_name as payoutBankName, pb.account_holder as payoutAccountHolder, pb.last4 as payoutBankLast4
		 FROM orders o 
		 LEFT JOIN coins c ON c.id = o.coin_id
		 LEFT JOIN deposit_routes dr ON dr.id = o.deposit_route_id
		 LEFT JOIN payment_cards pc ON pc.id = o.payout_card_id
		 LEFT JOIN payment_banks pb ON pb.id = o.payout_bank_id
		 WHERE ${where} ORDER BY o.created_at DESC LIMIT 200`, params);
    return res.json({ items: rows });
});
router.get('/orders', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const status = String(req.query.status || '').trim();
    const uid = String(req.query.uid || '').trim();
    const params = {};
    let where = '1=1';
    if (status) {
        params.status = status;
        where += ' AND o.status = :status';
    }
    if (uid) {
        const [u] = await pool.query(`SELECT id, owner_id FROM users WHERE uid = :uid LIMIT 1`, { uid });
        if (!u || !u.length)
            return res.json({ items: [] });
        params.targetUserId = Number(u[0].id);
        where += ' AND o.user_id = :targetUserId';
        if (actor.role === 2) {
            const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
            if (!me || !me.length)
                return res.status(401).json({ error: 'Unauthorized' });
            if (Number(u[0].owner_id) !== Number(me[0].id))
                return res.status(403).json({ error: 'Forbidden' });
        }
    }
    else if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length)
            return res.status(401).json({ error: 'Unauthorized' });
        params.ownerId = me[0].id;
        where += ' AND u.owner_id = :ownerId';
    }
    const [rows] = await pool.query(`SELECT o.id, o.user_id as userId, u.email as userEmail, u.first_name as firstName, u.last_name as lastName, o.type, o.coin_id as coinId, o.coin_symbol as coinSymbol,
		o.amount_coin as amountCoin, o.amount_fiat as amountFiat, o.currency, o.status, o.reject_reason as rejectReason, o.created_at as createdAt
		 FROM orders o JOIN users u ON u.id = o.user_id WHERE ${where} ORDER BY o.created_at DESC LIMIT 200`, params);
    return res.json({ items: rows });
});
router.post('/orders/:id/approve', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const id = Number(req.params.id);
    const conn = await pool.getConnection();
    try {
        const [rows] = await conn.query(`SELECT o.*, u.owner_id, u.uid as userUid FROM orders o JOIN users u ON u.id = o.user_id WHERE o.id = :id LIMIT 1`, { id });
        if (!rows || !rows.length)
            return res.status(404).json({ error: 'Not found' });
        const rec = rows[0];
        if (actor.role === 2) {
            const [me] = await conn.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
            if (!me || !me.length || Number(rec.owner_id) !== Number(me[0].id))
                return res.status(403).json({ error: 'Forbidden' });
        }
        await conn.beginTransaction();
        // ensure wallet exists
        const [wallet] = await conn.query(`SELECT id FROM user_wallets WHERE user_id = :uid AND coin_id = :coinId LIMIT 1`, { uid: rec.user_id, coinId: rec.coin_id });
        let walletId;
        if (!wallet || !wallet.length) {
            const [ins] = await conn.query(`INSERT INTO user_wallets (user_id, coin_id, balance, locked) VALUES (:uid, :coinId, 0, 0)`, { uid: rec.user_id, coinId: rec.coin_id });
            walletId = Number(ins.insertId);
        }
        else {
            walletId = Number(wallet[0].id);
        }
        if (rec.type === 'buy') {
            await conn.query(`INSERT INTO transactions (user_id, user_wallet_id, coin_id, tx_type, source, status, amount, fee, from_address, to_address, tx_hash, network, confirmations, created_by, meta, note)
				 VALUES (:userId, :walletId, :coinId, 'deposit', 'system', 'completed', :amount, 0, NULL, NULL, NULL, NULL, 0, NULL, JSON_OBJECT('orderId', :oId), 'Order approved')`, { userId: rec.user_id, walletId, coinId: rec.coin_id, amount: rec.amount_coin, oId: rec.id });
        }
        else {
            // sell: unlock funds and create completed withdrawal transaction
            await conn.query(`UPDATE user_wallets SET locked = locked - :amt WHERE user_id = :uid AND coin_id = :coinId AND locked >= :amt`, { amt: rec.amount_coin, uid: rec.user_id, coinId: rec.coin_id });
            await conn.query(`INSERT INTO transactions (user_id, user_wallet_id, coin_id, tx_type, source, status, amount, fee, from_address, to_address, tx_hash, network, confirmations, created_by, meta, note)
				 VALUES (:userId, :walletId, :coinId, 'withdrawal', 'system', 'completed', :amount, 0, NULL, NULL, NULL, NULL, 0, NULL, JSON_OBJECT('orderId', :oId), 'Order approved')`, { userId: rec.user_id, walletId, coinId: rec.coin_id, amount: rec.amount_coin, oId: rec.id });
        }
        await conn.query(`UPDATE orders SET status = 'approved' WHERE id = :id`, { id });
        await conn.commit();
        try {
            emitToUser(rec.userUid, 'order:approved', { id, type: rec.type, coinSymbol: rec.coin_symbol, amountCoin: rec.amount_coin });
        }
        catch { }
        return res.json({ ok: true });
    }
    catch (e) {
        try {
            await conn.rollback?.();
        }
        catch { }
        return res.status(500).json({ error: 'Failed to approve' });
    }
    finally {
        conn.release();
    }
});
router.post('/orders/:id/reject', authGuard, roleGuard([1, 2]), async (req, res) => {
    const actor = req.user;
    const id = Number(req.params.id);
    const reason = String(req.body?.reason || '').slice(0, 255) || null;
    const [rows] = await pool.query(`SELECT o.*, u.owner_id, u.uid as userUid FROM orders o JOIN users u ON u.id = o.user_id WHERE o.id = :id LIMIT 1`, { id });
    if (!rows || !rows.length)
        return res.status(404).json({ error: 'Not found' });
    const rec = rows[0];
    if (actor.role === 2) {
        const [me] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: actor.sub });
        if (!me || !me.length || Number(rec.owner_id) !== Number(me[0].id))
            return res.status(403).json({ error: 'Forbidden' });
    }
    if (rec.type === 'sell') {
        // Unlock funds on rejection
        await pool.query(`UPDATE user_wallets SET locked = locked - :amt WHERE user_id = :uid AND coin_id = :coinId AND locked >= :amt`, { amt: rec.amount_coin, uid: rec.user_id, coinId: rec.coin_id });
    }
    await pool.query(`UPDATE orders SET status = 'rejected', reject_reason = :reason WHERE id = :id`, { id, reason });
    try {
        emitToUser(rec.userUid, 'order:rejected', { id, reason });
    }
    catch { }
    return res.json({ ok: true });
});
export default router;
