import { Router } from 'express';
import pool from '../config/db.js';
import { authGuard } from '../middleware/auth.js';
import { z } from 'zod';
import { getIO, emitToUser } from '../socket.js';
import { getSimplePrices, getSimpleMarket } from '../services/coingecko.js';
const router = Router();
// GET /api/wallets - list current user's wallets with coin info
router.get('/', authGuard, async (req, res) => {
    const user = req.user;
    // Initialize wallets for active system coins if missing
    await pool.query(`INSERT INTO user_wallets (user_id, coin_id)
     SELECT u.id, c.id
     FROM users u CROSS JOIN coins c
     WHERE u.uid = :uid AND c.is_active = 1
     AND NOT EXISTS (
       SELECT 1 FROM user_wallets uw WHERE uw.user_id = u.id AND uw.coin_id = c.id
     )`, { uid: user.sub });
    // Fetch wallets + coin meta (including CoinGecko id/logo) and user's preferred currency
    const [rows] = await pool.query(`SELECT uw.id, uw.balance, uw.locked, c.id as coinId, c.symbol, c.name, c.decimals, c.coingecko_id as coingeckoId, c.logo_url as logoUrl,
            (SELECT JSON_UNQUOTE(JSON_EXTRACT(u.preferences, '$.currency')) FROM users u WHERE u.uid = :userId LIMIT 1) as prefCurrency
     FROM user_wallets uw
     JOIN coins c ON c.id = uw.coin_id
     WHERE uw.user_id = (SELECT id FROM users WHERE uid = :userId LIMIT 1)
     ORDER BY c.symbol`, { userId: user.sub });
    const vsCurrency = String(rows && rows.length ? rows[0].prefCurrency || 'USD' : 'USD').toUpperCase();
    const ids = Array.from(new Set((rows || []).map(r => (r.coingeckoId || '').toLowerCase()).filter(Boolean)));
    let pricesById = {};
    let marketById = {};
    if (ids.length > 0) {
        try {
            marketById = await getSimpleMarket(ids, vsCurrency, { ttlMs: 60000 });
            // fallback for any missing entries
            const missing = ids.filter(id => !marketById[id]);
            if (missing.length) {
                pricesById = await getSimplePrices(missing, vsCurrency, { ttlMs: 60000 });
            }
        }
        catch { }
    }
    res.json(rows.map(r => {
        const cgId = (r.coingeckoId || '').toLowerCase();
        const m = cgId ? marketById[cgId] : undefined;
        const price = m?.price ?? (cgId ? (pricesById[cgId] ?? null) : null);
        return {
            id: r.id,
            balance: r.balance,
            locked: r.locked,
            coin: { id: r.coinId, symbol: r.symbol, name: r.name, decimals: r.decimals },
            price,
            currency: vsCurrency,
            image: r.logoUrl || null,
            change24h: typeof m?.change24h === 'number' ? m.change24h : null,
            vol24h: typeof m?.vol24h === 'number' ? m.vol24h : null,
        };
    }));
});
// GET /api/wallets/transactions?limit=20&offset=0 - recent transactions across all wallets for current user
router.get('/transactions', authGuard, async (req, res) => {
    const user = req.user;
    const limit = Math.min(Number(req.query.limit ?? 20), 100);
    const offset = Math.max(Number(req.query.offset ?? 0), 0);
    const [rows] = await pool.query(`SELECT t.id, t.public_id as publicId, t.tx_type as txType, t.source, t.status, t.amount, t.fee, t.from_address as fromAddress, t.to_address as toAddress,
            t.tx_hash as txHash, t.network, t.confirmations, t.created_at as createdAt,
            c.symbol, t.note
     FROM transactions t
     JOIN users u ON u.id = t.user_id
     LEFT JOIN coins c ON c.id = t.coin_id
     WHERE u.uid = :uid
       AND (t.meta IS NULL OR JSON_UNQUOTE(JSON_EXTRACT(t.meta, '$.hidden')) IS NULL OR JSON_UNQUOTE(JSON_EXTRACT(t.meta, '$.hidden')) = 'false')
     ORDER BY t.id DESC
     LIMIT :limit OFFSET :offset`, { uid: user.sub, limit, offset });
    const items = rows.map(r => ({ ...r, note: r.note ?? null }));
    res.json({ items, limit, offset });
});
// New: portfolio history (per-day coin balances for last N days)
router.get('/portfolio-history', authGuard, async (req, res) => {
    const user = req.user;
    const days = Math.min(Math.max(Number(req.query.days ?? 90), 1), 365);
    // Resolve user id
    const [u] = await pool.query(`SELECT id FROM users WHERE uid = :uid LIMIT 1`, { uid: user.sub });
    if (!u || !u.length)
        return res.status(401).json({ error: 'Unauthorized' });
    const userId = Number(u[0].id);
    // Daily net changes per coin in the period (completed transactions only)
    const [txRows] = await pool.query(`SELECT DATE(t.created_at) as d, c.symbol as symbol,
				SUM(CASE WHEN t.status = 'completed' AND t.tx_type IN ('deposit','adjustment','transfer') THEN t.amount
					WHEN t.status = 'completed' AND t.tx_type = 'withdrawal' THEN -t.amount ELSE 0 END) as net
		 FROM transactions t
		 JOIN coins c ON c.id = t.coin_id
		 WHERE t.user_id = :uid
		   AND t.created_at >= DATE_SUB(CURDATE(), INTERVAL :days DAY)
		   AND (t.meta IS NULL OR JSON_UNQUOTE(JSON_EXTRACT(t.meta, '$.hidden')) IS NULL OR JSON_UNQUOTE(JSON_EXTRACT(t.meta, '$.hidden')) = 'false')
		 GROUP BY d, symbol
		 ORDER BY d ASC`, { uid: userId, days });
    // Build date list oldest->today
    const dates = [];
    for (let i = days - 1; i >= 0; i--) {
        const d = new Date();
        d.setDate(d.getDate() - i);
        dates.push(d.toISOString().slice(0, 10));
    }
    // Map changes per date/symbol
    const netByDateSym = {};
    for (const r of (txRows || [])) {
        const d = String(r.d);
        const sym = String(r.symbol || '').toUpperCase();
        if (!netByDateSym[d])
            netByDateSym[d] = {};
        netByDateSym[d][sym] = (Number(r.net) || 0);
    }
    // Accumulate from zero forward so days before first tx remain 0
    let prev = {};
    const series = [];
    let hasNonZero = false;
    for (const d of dates) {
        const chg = netByDateSym[d] || {};
        const next = { ...prev };
        for (const sym of Object.keys(chg))
            next[sym] = (next[sym] || 0) + (chg[sym] || 0);
        // detect any non-zero balances
        for (const v of Object.values(next)) {
            if (Math.abs(Number(v) || 0) > 1e-12) {
                hasNonZero = true;
                break;
            }
        }
        series.push({ date: d, balances: next });
        prev = next;
    }
    // If the window has no non-zero balances at all but user currently has balances, set last day to current balances
    if (!hasNonZero) {
        const [wbals] = await pool.query(`SELECT c.symbol as symbol, w.balance as balance FROM user_wallets w JOIN coins c ON c.id = w.coin_id WHERE w.user_id = :uid`, { uid: userId });
        const curr = {};
        for (const r of (wbals || []))
            curr[String(r.symbol).toUpperCase()] = Number(r.balance) || 0;
        if (Object.keys(curr).length > 0 && series.length) {
            series[series.length - 1] = { date: series[series.length - 1].date, balances: curr };
        }
    }
    return res.json({ items: series });
});
// POST /api/wallets/:id/withdraw - create a pending withdrawal request
router.post('/:id/withdraw', authGuard, async (req, res) => {
    const user = req.user;
    const walletId = Number(req.params.id);
    const schema = z.object({
        amount: z.string().min(1),
        toAddress: z.string().min(6),
        tagMemo: z.string().optional(),
        network: z.string().optional(),
    });
    const parsed = schema.safeParse(req.body);
    if (!parsed.success)
        return res.status(400).json({ error: 'Invalid input' });
    const data = parsed.data;
    const conn = await pool.getConnection();
    try {
        const [owner] = await conn.query(`SELECT uw.id, uw.balance, uw.locked, uw.user_id, c.id as coinId, c.symbol
       FROM user_wallets uw JOIN users u ON u.id = uw.user_id JOIN coins c ON c.id = uw.coin_id
       WHERE uw.id = :walletId AND u.uid = :uid LIMIT 1`, { walletId, uid: user.sub });
        if (!owner || !owner.length)
            return res.status(404).json({ error: 'Wallet not found' });
        const rec = owner[0];
        const amountNum = Number(data.amount);
        if (!isFinite(amountNum) || amountNum <= 0)
            return res.status(400).json({ error: 'Invalid amount' });
        const available = Number(rec.balance) - Number(rec.locked || 0);
        if (amountNum > available)
            return res.status(400).json({ error: 'Insufficient available balance' });
        const meta = JSON.stringify({ requestedAt: new Date().toISOString(), tagMemo: data.tagMemo || undefined });
        const [result] = await conn.query(`INSERT INTO transactions (user_id, user_wallet_id, coin_id, tx_type, source, status, amount, fee, from_address, to_address, tx_hash, network, confirmations, created_by, meta)
       VALUES (:userId, :walletId, :coinId, 'withdrawal', 'system', 'pending', :amount, 0, NULL, :toAddress, NULL, :network, 0, NULL, :meta)`, { userId: rec.user_id, walletId, coinId: rec.coinId, amount: data.amount, toAddress: data.toAddress, network: data.network || null, meta });
        // Resolve owner agent and admins
        const [who] = await conn.query(`SELECT u.owner_id as ownerId, (SELECT uid FROM users WHERE id = u.owner_id) as ownerUid FROM users u WHERE u.id = :id LIMIT 1`, { id: rec.user_id });
        const ownerUid = who && who.length && who[0].ownerUid ? String(who[0].ownerUid) : null;
        // Notify owning agent; if no owner, notify all admins
        try {
            if (ownerUid) {
                emitToUser(ownerUid, 'transaction:new-request', { id: result.insertId, type: 'withdrawal', amount: data.amount, currency: rec.symbol, userId: rec.user_id, targetUid: ownerUid });
            }
            else {
                getIO().to('role:admin').emit('transaction:new-request', { id: result.insertId, type: 'withdrawal', amount: data.amount, currency: rec.symbol, userId: rec.user_id, targetRole: 'admin' });
            }
        }
        catch { }
        try {
            const [emailRow] = await conn.query(`SELECT email, first_name FROM users WHERE id = :id LIMIT 1`, { id: rec.user_id });
            if (emailRow && emailRow.length) {
                const email = String(emailRow[0].email);
                const firstName = String(emailRow[0].first_name || 'there');
                const { getEmailService } = await import('../services/email/index.js');
                await getEmailService().sendWithdrawalRequested(email, { userName: firstName, amount: data.amount, asset: rec.symbol, network: data.network || null, address: data.toAddress, requestId: result.insertId });
            }
        }
        catch { }
        return res.status(201).json({ id: result.insertId });
    }
    finally {
        conn.release();
    }
});
// GET /api/wallets/:id/addresses - active deposit addresses for this wallet
router.get('/:id/addresses', authGuard, async (req, res) => {
    const user = req.user;
    const walletId = req.params.id;
    // Ownership check
    const [own] = await pool.query(`SELECT id FROM user_wallets WHERE id = :id AND user_id = (SELECT id FROM users WHERE uid = :userId LIMIT 1) LIMIT 1`, { id: walletId, userId: user.sub });
    if (!own || own.length === 0)
        return res.status(404).json({ error: 'Not found' });
    const [rows] = await pool.query(`SELECT id, address, tag_memo as tagMemo, kind, is_active as isActive, created_at as createdAt
     FROM wallet_addresses WHERE user_wallet_id = :id AND is_active = 1 AND kind = 'deposit'`, { id: walletId });
    res.json(rows);
});
// GET /api/wallets/:id/transactions?limit=20&offset=0
router.get('/:id/transactions', authGuard, async (req, res) => {
    const user = req.user;
    const walletId = req.params.id;
    const limit = Math.min(Number(req.query.limit ?? 20), 100);
    const offset = Math.max(Number(req.query.offset ?? 0), 0);
    const [own] = await pool.query(`SELECT id FROM user_wallets WHERE id = :id AND user_id = (SELECT id FROM users WHERE uid = :userId LIMIT 1) LIMIT 1`, { id: walletId, userId: user.sub });
    if (!own || own.length === 0)
        return res.status(404).json({ error: 'Not found' });
    // Total count for pagination
    const [countRows] = await pool.query(`SELECT COUNT(*) AS cnt
     FROM transactions
     WHERE user_wallet_id = :id AND user_id = (SELECT id FROM users WHERE uid = :userId LIMIT 1)
       AND (meta IS NULL OR JSON_UNQUOTE(JSON_EXTRACT(meta, '$.hidden')) IS NULL OR JSON_UNQUOTE(JSON_EXTRACT(meta, '$.hidden')) = 'false')`, { id: walletId, userId: user.sub });
    const total = (countRows && countRows.length) ? Number(countRows[0].cnt) : 0;
    const [rows] = await pool.query(`SELECT t.id, t.public_id as publicId, t.tx_type as txType, t.source, t.status, t.amount, t.fee, t.from_address as fromAddress, t.to_address as toAddress,
            t.tx_hash as txHash, t.network, t.confirmations, t.created_at as createdAt, t.note,
            COALESCE(c.symbol, 'UNKNOWN') as symbol
     FROM transactions t
     LEFT JOIN coins c ON c.id = t.coin_id
     WHERE t.user_wallet_id = :id AND t.user_id = (SELECT id FROM users WHERE uid = :userId LIMIT 1)
       AND (t.meta IS NULL OR JSON_UNQUOTE(JSON_EXTRACT(t.meta, '$.hidden')) IS NULL OR JSON_UNQUOTE(JSON_EXTRACT(t.meta, '$.hidden')) = 'false')
     ORDER BY t.id DESC
     LIMIT :limit OFFSET :offset`, { id: walletId, userId: user.sub, limit, offset });
    const items = rows.map(r => ({ ...r, note: r.note ?? null }));
    res.json({ items, limit, offset, total });
});
// New: GET /api/wallets/address?symbol=BTC or ?coinId=1 - user's active deposit address for a coin
router.get('/address', authGuard, async (req, res) => {
    const user = req.user;
    const symbolRaw = String(req.query.symbol ?? '').trim();
    const coinIdRaw = req.query.coinId;
    const hasSymbol = symbolRaw.length > 0;
    const hasCoinId = coinIdRaw != null && String(coinIdRaw).length > 0 && !isNaN(Number(coinIdRaw));
    if (!hasSymbol && !hasCoinId)
        return res.status(400).json({ error: 'symbol or coinId is required' });
    const params = { uid: user.sub };
    let where = '';
    if (hasSymbol) {
        where = 'UPPER(c.symbol) = :symbol';
        params.symbol = symbolRaw.toUpperCase();
    }
    else {
        where = 'c.id = :coinId';
        params.coinId = Number(coinIdRaw);
    }
    const [w] = await pool.query(`SELECT uw.id as walletId, c.id as coinId, c.symbol as symbol
     FROM user_wallets uw
     JOIN users u ON u.id = uw.user_id
     JOIN coins c ON c.id = uw.coin_id
     WHERE u.uid = :uid AND ${where}
     LIMIT 1`, params);
    if (!w || !w.length)
        return res.status(404).json({ error: 'Wallet not found' });
    const walletId = Number(w[0].walletId);
    const [addr] = await pool.query(`SELECT id, address, tag_memo as tagMemo, kind, is_active as isActive, created_at as createdAt
     FROM wallet_addresses
     WHERE user_wallet_id = :id AND kind = 'deposit' AND is_active = 1
     ORDER BY id DESC
     LIMIT 1`, { id: walletId });
    const address = addr && addr.length ? addr[0] : null;
    return res.json({
        walletId,
        coin: { id: Number(w[0].coinId), symbol: String(w[0].symbol) },
        address
    });
});
export default router;
