import { Server as SocketIOServer } from 'socket.io';
import cookie from 'cookie';
import { verifyJwt } from './utils/jwt.js';
import pool from './config/db.js';
import { ensureVsFeed, releaseVsFeed, getLatestSnapshot } from './services/marketFeed.js';
let io = null;
const onlineUserIds = new Set();
export function initIO(server, origin) {
    io = new SocketIOServer(server, { cors: { origin, credentials: true } });
    io.on('connection', async (socket) => {
        let joinedVs = null;
        try {
            const cookies = socket.handshake.headers.cookie ? cookie.parse(String(socket.handshake.headers.cookie)) : {};
            const token = cookies['token'];
            const payload = token ? verifyJwt(token) : null;
            if (payload?.sub) {
                const uid = String(payload.sub);
                const role = Number(payload.role || 3);
                onlineUserIds.add(uid);
                socket.data.userId = uid;
                socket.data.role = role;
                // Join per-user room for targeted emits
                socket.join(`user:${uid}`);
                if (role === 1)
                    socket.join('role:admin');
                if (role === 2)
                    socket.join('role:agent');
                io.emit('presence:update', { userId: uid, online: true });
                // Determine user's preferred currency from DB
                const [prefRows] = await pool.query(`SELECT JSON_UNQUOTE(JSON_EXTRACT(u.preferences, '$.currency')) as currency FROM users u WHERE u.uid = :uid LIMIT 1`, { uid });
                const vs = String(prefRows && prefRows.length ? prefRows[0].currency : 'USD').toUpperCase();
                joinedVs = vs;
                socket.join(`market:${vs}`);
                ensureVsFeed(vs);
                const snap = getLatestSnapshot(vs);
                if (snap)
                    socket.emit('rates:snapshot', snap);
            }
        }
        catch { }
        socket.on('market:resubscribe', async () => {
            try {
                const uid = socket.data.userId;
                if (!uid)
                    return;
                // Read fresh preference
                const [prefRows] = await pool.query(`SELECT JSON_UNQUOTE(JSON_EXTRACT(u.preferences, '$.currency')) as currency FROM users u WHERE u.uid = :uid LIMIT 1`, { uid });
                const nextVs = String(prefRows && prefRows.length ? prefRows[0].currency || 'USD' : 'USD').toUpperCase();
                if (joinedVs && joinedVs !== nextVs) {
                    socket.leave(`market:${joinedVs}`);
                    releaseVsFeed(joinedVs);
                    joinedVs = nextVs;
                    socket.join(`market:${nextVs}`);
                    ensureVsFeed(nextVs);
                }
                else if (!joinedVs) {
                    joinedVs = nextVs;
                    socket.join(`market:${nextVs}`);
                    ensureVsFeed(nextVs);
                }
                const snap = getLatestSnapshot(nextVs);
                if (snap)
                    socket.emit('rates:snapshot', snap);
            }
            catch { }
        });
        socket.on('disconnect', () => {
            const uid = socket.data.userId;
            if (uid && onlineUserIds.has(uid)) {
                onlineUserIds.delete(uid);
                io.emit('presence:update', { userId: uid, online: false });
            }
            if (joinedVs) {
                releaseVsFeed(joinedVs);
                joinedVs = null;
            }
        });
    });
    return io;
}
export function getIO() {
    if (!io) {
        throw new Error('Socket.io not initialized');
    }
    return io;
}
export function isUserOnline(userId) {
    return onlineUserIds.has(String(userId));
}
export function getOnlineUserIds() {
    return Array.from(onlineUserIds);
}
// Targeted emit helpers
export function emitToUser(userUid, event, payload) {
    if (!io)
        return;
    io.to(`user:${userUid}`).emit(event, payload);
}
